<?php
require_once 'config.php';

$screenId = isset($_GET['screen_id']) ? (int)$_GET['screen_id'] : 0;
if (!$screenId) {
    die('Falta screen_id en la URL. Ej: player.php?screen_id=1');
}

$stmt = $pdo->prepare("SELECT * FROM screens WHERE id=?");
$stmt->execute([$screenId]);
$screen = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$screen) {
    die('Pantalla no encontrada');
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title><?php echo htmlspecialchars($screen['name']); ?> - Player PRISA</title>
    <meta name="viewport" content="width=device-width, initial-scale=1, viewport-fit=cover">
    <style>
        html, body {
            margin: 0;
            padding: 0;
            width: 100%;
            height: 100%;
            background: #000;
            overflow: hidden;
            font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
            color: #fff;
        }
        #playerStage {
            position: fixed;
            inset: 0;
            display: flex;
            align-items: center;
            justify-content: center;
            background: #000;
        }
        .player-image,
        .player-video {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
    </style>
</head>
<body>
<div id="playerStage"></div>

<script>
const screenId = <?php echo $screenId; ?>;
let playlist = [];
let currentIndex = 0;
let lastUpdate = 0;
let currentTimeout = null;
let isPlayingVideo = false;

const CACHE_KEY = 'prisa_playlist_screen_' + screenId;

function logStatus(msg) {
    console.log('[PRISA PLAYER]', msg);
}

function loadFromCache() {
    try {
        const raw = localStorage.getItem(CACHE_KEY);
        if (!raw) return false;
        const data = JSON.parse(raw);
        playlist = data.items || [];
        lastUpdate = data.last_update || 0;
        if (playlist.length) {
            logStatus('Reproduciendo playlist guardada (' + playlist.length + ' piezas, modo sin conexión)');
            return true;
        }
    } catch (e) {
        console.error('Error leyendo cache local', e);
    }
    return false;
}

function saveToCache() {
    try {
        const data = {
            items: playlist,
            last_update: lastUpdate
        };
        localStorage.setItem(CACHE_KEY, JSON.stringify(data));
    } catch (e) {
        console.error('Error guardando cache local', e);
    }
}

async function fetchPlaylist(force = false) {
    try {
        const url = 'api_playlist.php?screen_id=' + screenId + '&since=' + (force ? 0 : lastUpdate);
        const res = await fetch(url + '&_=' + Date.now(), { cache: 'no-store' });
        const data = await res.json();

        if (data.changed) {
            playlist = data.items || [];
            lastUpdate = data.last_update || 0;
            if (playlist.length) {
                logStatus('Playlist activa (' + playlist.length + ' piezas)');
                saveToCache();
            } else {
                logStatus('Sin creatividades activas en CMS.');
            }
        }
        return true;
    } catch (e) {
        console.error('Error consultando playlist online', e);
        logStatus('Sin conexión, usando última playlist guardada (si existe)');
        if (!playlist.length) {
            loadFromCache();
        }
        return false;
    }
}

function clearStage() {
    const stage = document.getElementById('playerStage');
    stage.innerHTML = '';
}

function playNext() {
    if (!playlist.length) {
        clearStage();
        logStatus('Sin creatividades (online y cache vacíos). Reintentando...');
        currentTimeout = setTimeout(() => {
            currentTimeout = null;
            playNext();
        }, 5000);
        return;
    }

    if (currentIndex >= playlist.length) {
        currentIndex = 0;
    }

    const item = playlist[currentIndex];
    currentIndex++;

    clearStage();
    const stage = document.getElementById('playerStage');

    if (item.type === 'video') {
        const video = document.createElement('video');
        video.src = item.url + (item.url.includes('?') ? '&' : '?') + '_=' + Date.now();
        video.autoplay = true;
        video.muted = false;
        video.playsInline = true;
        video.className = 'player-video';
        stage.appendChild(video);
        isPlayingVideo = true;

        let ended = false;
        const next = () => {
            if (ended) return;
            ended = true;
            isPlayingVideo = false;
            currentTimeout = null;
            playNext();
        };

        video.addEventListener('ended', next);
        video.addEventListener('error', next);

        const maxDuration = (item.duration && item.duration > 0 ? item.duration : 30) * 1000;
        currentTimeout = setTimeout(next, maxDuration);

    } else {
        const img = document.createElement('img');
        img.src = item.url + (item.url.includes('?') ? '&' : '?') + '_=' + Date.now();
        img.className = 'player-image';
        stage.appendChild(img);
        isPlayingVideo = false;

        const dur = (item.duration && item.duration > 0 ? item.duration : 10) * 1000;
        currentTimeout = setTimeout(() => {
            currentTimeout = null;
            playNext();
        }, dur);
    }
}

async function loop() {
    await fetchPlaylist(false);
    if (!isPlayingVideo && !currentTimeout) {
        playNext();
    }
}

(async () => {
    const ok = await fetchPlaylist(true);
    if (!ok && !playlist.length) {
        loadFromCache();
    }
    playNext();
    setInterval(loop, 20000);
})();
</script>
</body>
</html>
